#include <windows.h>
#include <atlbase.h>
#include <atlwin.h>
#include <atlapp.h>
CAppModule _Module;
#include <stdio.h>
#include <stdlib.h>
#include <atlctrls.h>
#include <atlframe.h>
#include <atlcrack.h>
#include <atldlgs.h>
#include <atlstr.h>
#include <atltypes.h>
#include <stdlib.h>
#include "explore.h"
#include "dpi.h"
#include "graph.h"
#include "resource.h"

#define MSG_CAMERA			(WM_APP + 1)
#define TIMER_EPSILON		10
#define TIMER_ID			1
#define TIMER_TEMP			2
#define ROW_MIN				1
#define ROW_MAX				20
#define COL_MIN				1
#define COL_MAX				20
#define AREA_MIN			3
#define AREA_MAX			100
#define TEMP_MIN			0
#define TEMP_MAX			3600

CDPI g_dpi;

typedef struct {
	UINT	delayTime;
	UINT	expoTime;
	USHORT	expoGain;
} Expo; /* exposure parameter */

static CString FormatExpoTime(DWORD dwExpoTime)
{
	CString str;
	if (0 == dwExpoTime % 1000)
		str.Format(L"%u", dwExpoTime / 1000);
	else if (0 == dwExpoTime % 100)
		str.Format(L"%u.%u", dwExpoTime / 1000, (dwExpoTime % 1000) / 100);
	else if (0 == dwExpoTime % 10)
		str.Format(L"%u.%02u", dwExpoTime / 1000, (dwExpoTime % 1000) / 10);
	else
		str.Format(L"%u.%03u", dwExpoTime / 1000, dwExpoTime % 1000);
	return str;
}

static CString FormatString(const wchar_t* szFormat, ...)
{
	CString str;
	va_list valist;
	va_start(valist, szFormat);
	str.FormatV(szFormat, valist);
	va_end(valist);
	return str;
}

static bool GetDlgInt(CWindow* pDlg, UINT nID, DWORD& val, DWORD minval, DWORD maxval)
{
	BOOL bTrans = FALSE;
	val = pDlg->GetDlgItemInt(nID, &bTrans, FALSE);
	if (!bTrans)
	{
		pDlg->GotoDlgCtrl(pDlg->GetDlgItem(nID));
		AtlMessageBox(pDlg->m_hWnd, L"Format error.", (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
		return true;
	}
	if ((val < minval) || (val > maxval))
	{
		pDlg->GotoDlgCtrl(pDlg->GetDlgItem(nID));
		AtlMessageBox(pDlg->m_hWnd, (LPCTSTR)FormatString(L"Out of range [%u, %u].", minval, maxval), (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
		return true;
	}

	return false; // everything ok
}

bool GetExpoTime(CWindow* pDlg, UINT nID, DWORD& expoTime, DWORD minExpoTime, DWORD maxExpoTime)
{
	CString str;
	pDlg->GetDlgItemText(nID, str);
	wchar_t* endptr = nullptr;
	const double d = wcstod((LPCTSTR)str, &endptr);
	if (endptr && (*endptr))
	{
		pDlg->GotoDlgCtrl(pDlg->GetDlgItem(nID));
		AtlMessageBox(pDlg->m_hWnd, L"Format error.", (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
		return true;
	}
	expoTime = (DWORD)(d * 1000 + 0.5);
	if ((expoTime < minExpoTime) || (expoTime > maxExpoTime))
	{
		pDlg->GotoDlgCtrl(pDlg->GetDlgItem(nID));
		AtlMessageBox(pDlg->m_hWnd, (LPCTSTR)FormatString(L"Out of range [%s, %s].", (LPCTSTR)FormatExpoTime(minExpoTime), (LPCTSTR)FormatExpoTime(maxExpoTime)), (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
		return true;
	}

	return false; // everything ok
}

class CExposureDlg : public CDialogImpl<CExposureDlg>
{
	friend class CConfigDlg;
	const unsigned*			m_expoTimeRange;
	const unsigned short*	m_expoGainRange;
	DWORD	m_expoTime, m_expoGain, m_delayTime;
public:
	enum { IDD = IDD_EXPOSURE };
	CExposureDlg(const unsigned* expoTimeRange, const unsigned short* expoGainRange)
	: m_expoTimeRange(expoTimeRange), m_expoGainRange(expoGainRange), m_expoTime(0), m_expoGain(EXPLORE_EXPOGAIN_MIN), m_delayTime(0)
	{
	}

	CExposureDlg(const unsigned* expoTimeRange, const unsigned short* expoGainRange, const Expo& expo)
	: m_expoTimeRange(expoTimeRange), m_expoGainRange(expoGainRange), m_expoTime(expo.expoTime), m_expoGain(expo.expoGain), m_delayTime(expo.delayTime)
	{
	}

	BEGIN_MSG_MAP(CExposureDlg)
		MESSAGE_HANDLER(WM_INITDIALOG, OnInitDialog)
		COMMAND_HANDLER(IDOK, BN_CLICKED, OnOK)
		COMMAND_HANDLER(IDCANCEL, BN_CLICKED, OnCancel)
	END_MSG_MAP()
private:
	LRESULT OnInitDialog(UINT /*uMsg*/, WPARAM /*wParam*/, LPARAM /*lParam*/, BOOL& /*bHandled*/)
	{
		CenterWindow(GetParent());

		SetDlgItemText(IDC_STATIC1, (LPCTSTR)FormatString(L"Range: [%s, %s]ms", (LPCTSTR)FormatExpoTime(m_expoTimeRange[0]), (LPCTSTR)FormatExpoTime(m_expoTimeRange[1])));
		SetDlgItemText(IDC_STATIC2, (LPCTSTR)FormatString(L"Range: [%hu, %hu]", m_expoGainRange[0], m_expoGainRange[1]));

		if (m_expoTime)
			SetDlgItemText(IDC_EDIT1, (LPCTSTR)FormatExpoTime(m_expoTime));
		SetDlgItemInt(IDC_EDIT2, m_expoGain, FALSE);
		SetDlgItemInt(IDC_EDIT3, m_delayTime, FALSE);
		return TRUE;
	}

	LRESULT OnOK(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (GetDlgInt(this, IDC_EDIT2, m_expoGain, m_expoGainRange[0], m_expoGainRange[1])
			|| GetDlgInt(this, IDC_EDIT3, m_delayTime, 0, UINT_MAX)
			|| GetExpoTime(this, IDC_EDIT1, m_expoTime, m_expoTimeRange[0], m_expoTimeRange[1]))
			return 0;
		EndDialog(wID);
		return 0;
	}

	LRESULT OnCancel(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		EndDialog(wID);
		return 0;
	}
};

class CConfigDlg : public CDialogImpl<CConfigDlg>
{
	friend class CMainFrame;
	HExplorecam	m_hCam;
	DWORD		m_row, m_col, m_area, m_bin, m_temp, m_scale;
	CRegKey		m_regkey;
	unsigned		m_expoTimeRange[2];
	unsigned short	m_expoGainRange[2];
	std::vector<Expo>	m_vecExpo;
public:
	enum { IDD = IDD_CONFIG };
	CConfigDlg(HExplorecam hCam)
		: m_hCam(hCam), m_row(10), m_col(10), m_area(5), m_bin(1), m_temp(0), m_scale(0)
	{
		m_regkey.Create(HKEY_CURRENT_USER, L"Software\\democns");
		m_regkey.QueryDWORDValue(L"row", m_row);
		m_regkey.QueryDWORDValue(L"col", m_col);
		m_regkey.QueryDWORDValue(L"area", m_area);

		DWORD dwLength = 0;
		m_regkey.QueryBinaryValue(L"expo", nullptr, &dwLength);
		if (dwLength && (dwLength % sizeof(Expo) == 0))
		{
			m_vecExpo.resize(dwLength / sizeof(Expo));
			m_regkey.QueryBinaryValue(L"expo", &m_vecExpo[0], &dwLength);
		}
		
		Explorecam_get_ExpTimeRange(m_hCam, &m_expoTimeRange[0], &m_expoTimeRange[1], nullptr);		
		Explorecam_get_ExpoAGainRange(m_hCam, &m_expoGainRange[0], &m_expoGainRange[1], nullptr);
	}

	BEGIN_MSG_MAP(CMainDlg)
		MESSAGE_HANDLER(WM_INITDIALOG, OnInitDialog)
		COMMAND_HANDLER(IDOK, BN_CLICKED, OnOK)
		COMMAND_HANDLER(IDCANCEL, BN_CLICKED, OnCancel)
		COMMAND_HANDLER(IDC_BUTTON1, BN_CLICKED, OnAdd)
		COMMAND_HANDLER(IDC_BUTTON2, BN_CLICKED, OnDelete)
		COMMAND_HANDLER(IDC_CHECK1, BN_CLICKED, OnCheck1)
		NOTIFY_HANDLER(IDC_LIST1, LVN_ITEMCHANGED, OnLvnItemchanged)
		NOTIFY_HANDLER(IDC_LIST1, NM_DBLCLK, OnNmDblclk)
	END_MSG_MAP()
private:
	LRESULT OnInitDialog(UINT /*uMsg*/, WPARAM /*wParam*/, LPARAM /*lParam*/, BOOL& /*bHandled*/)
	{
		CenterWindow(GetParent());

		{
			CComboBox box(GetDlgItem(IDC_COMBO1));
			box.AddString(L"Trigger");
			box.AddString(L"Live");
			DWORD dwValue = 0;
			m_regkey.QueryDWORDValue(L"trigger", dwValue);
			box.SetCurSel(dwValue);
		}
		{
			CComboBox box(GetDlgItem(IDC_COMBO2));
			box.AddString(L"Unsaturated Add");
			box.AddString(L"Saturating Add");
			box.AddString(L"Average");
			DWORD dwValue = 0;
			m_regkey.QueryDWORDValue(L"binmethod", dwValue);
			box.SetCurSel(dwValue);
		}
		if (E_NOTIMPL == Explorecam_get_Option(m_hCam, EXPLORE_OPTION_BITDEPTH, nullptr)) // support bitdepth
		{
			GetDlgItem(IDC_CHECK1).EnableWindow(FALSE);
			GetDlgItem(IDC_CHECK2).EnableWindow(FALSE);
		}
		else
		{
			DWORD dwValue = 1;
			m_regkey.QueryDWORDValue(L"bitdepth", dwValue);
			CheckDlgButton(IDC_CHECK1, dwValue ? 1 : 0);
			m_regkey.QueryDWORDValue(L"scale", m_scale);
			CheckDlgButton(IDC_CHECK2, m_scale ? 1 : 0);
		}
		if (E_NOTIMPL == Explorecam_get_Temperature(m_hCam, nullptr)) // support get the temperature of the sensor
			GetDlgItem(IDC_EDIT5).EnableWindow(FALSE);
		else
		{
			CUpDownCtrl ctrl(GetDlgItem(IDC_SPIN5));
			ctrl.SetRange(TEMP_MIN, TEMP_MAX);
			m_regkey.QueryDWORDValue(L"temp", m_temp);
			SetDlgItemInt(IDC_EDIT5, m_temp, FALSE);
		}
		{
			CUpDownCtrl ctrl(GetDlgItem(IDC_SPIN1));
			ctrl.SetRange(ROW_MIN, ROW_MAX);
			SetDlgItemInt(IDC_EDIT1, m_row);
		}
		{
			CUpDownCtrl ctrl(GetDlgItem(IDC_SPIN2));
			ctrl.SetRange(COL_MIN, COL_MAX);
			SetDlgItemInt(IDC_EDIT2, m_col);
		}
		{
			CUpDownCtrl ctrl(GetDlgItem(IDC_SPIN3));
			ctrl.SetRange(AREA_MIN, AREA_MAX);
			SetDlgItemInt(IDC_EDIT3, m_area);
		}
		{
			CUpDownCtrl ctrl(GetDlgItem(IDC_SPIN4));
			ctrl.SetRange(1, 8);
			DWORD dwValue = 1;
			m_regkey.QueryDWORDValue(L"binvalue", dwValue);
			SetDlgItemInt(IDC_EDIT4, dwValue);
		}

		{
			CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
			ctrl.SetExtendedListViewStyle(ctrl.GetExtendedListViewStyle() | LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES);
			ctrl.AddColumn(L"Time(ms)", 0);
			ctrl.AddColumn(L"Gain", 1);
			ctrl.AddColumn(L"Delay(ms)", 2);
			CRect rect;
			ctrl.GetClientRect(&rect);
			const int width = rect.Width() - GetSystemMetrics(SM_CXVSCROLL) - 8;
			ctrl.SetColumnWidth(0, width / 3);
			ctrl.SetColumnWidth(1, width / 3);
			ctrl.SetColumnWidth(2, width / 3);
			for (size_t i = 0; i < m_vecExpo.size(); ++i)
			{
				ctrl.AddItem(i, 0, (LPCTSTR)FormatExpoTime(m_vecExpo[i].expoTime));
				ctrl.SetItemText(i, 1, (LPCTSTR)FormatString(L"%hu", m_vecExpo[i].expoGain));
				ctrl.SetItemText(i, 2, (LPCTSTR)FormatString(L"%u", m_vecExpo[i].delayTime));
			}
		}

		GetDlgItem(IDC_BUTTON2).EnableWindow(FALSE);
		GetDlgItem(IDOK).EnableWindow(!m_vecExpo.empty());
		return TRUE;
	}

	LRESULT OnOK(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (m_vecExpo.empty())
			return 0;
		for (size_t i = 0; i < m_vecExpo.size(); ++i)
		{
			if ((m_vecExpo[i].expoTime < m_expoTimeRange[0]) || (m_vecExpo[i].expoTime > m_expoTimeRange[1]))
			{
				AtlMessageBox(m_hWnd, (LPCTSTR)FormatString(L"Exposure time out of range [%s, %s].", (LPCTSTR)FormatExpoTime(m_expoTimeRange[0]), (LPCTSTR)FormatExpoTime(m_expoTimeRange[1])), (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
				return 0;
			}
			if ((m_vecExpo[i].expoGain < m_expoGainRange[0]) || (m_vecExpo[i].expoGain > m_expoGainRange[1]))
			{
				AtlMessageBox(m_hWnd, (LPCTSTR)FormatString(L"Exposure gain out of range [%hu, %hu].", m_expoGainRange[0], m_expoGainRange[1]), (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
				return 0;
			}
		}

		if (GetDlgInt(this, IDC_EDIT1, m_row, ROW_MIN, ROW_MAX) || GetDlgInt(this, IDC_EDIT2, m_col, COL_MIN, COL_MAX)
			|| GetDlgInt(this, IDC_EDIT3, m_area, AREA_MIN, AREA_MAX)
			|| GetDlgInt(this, IDC_EDIT4, m_bin, 1, 8))
			return 0;
		if (E_NOTIMPL != Explorecam_get_Temperature(m_hCam, nullptr))
		{
			if (GetDlgInt(this, IDC_EDIT5, m_temp, TEMP_MIN, TEMP_MAX))
				return 0;
			m_regkey.SetDWORDValue(L"temp", m_temp);
		}

		if (m_bin > 1)
		{
			CComboBox box(GetDlgItem(IDC_COMBO2));
			m_regkey.SetDWORDValue(L"binmethod", box.GetCurSel());
			switch (box.GetCurSel())
			{
			case 1:
				break;
			case 2:
				m_bin |= 0x80;
				break;
			default:
				m_bin |= 0x40;
				break;
			}
			Explorecam_put_Option(m_hCam, EXPLORE_OPTION_BINNING, m_bin);
		}
		{
			int width = 0, height = 0;
			Explorecam_get_FinalSize(m_hCam, &width, &height);
			if ((width / m_col < m_area) || (height / m_row < m_area))
			{
				AtlMessageBox(m_hWnd, L"Image size overflow.", (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
				return 0;
			}
		}

		m_regkey.SetDWORDValue(L"row", m_row);
		m_regkey.SetDWORDValue(L"col", m_col);
		m_regkey.SetDWORDValue(L"area", m_area);
		m_regkey.SetBinaryValue(L"expo", &m_vecExpo[0], sizeof(Expo) * m_vecExpo.size());
		m_regkey.SetDWORDValue(L"binvalue", m_bin);

		if (E_NOTIMPL != Explorecam_get_Option(m_hCam, EXPLORE_OPTION_BITDEPTH, nullptr)) // support bitdepth
		{
			const int bCheck = IsDlgButtonChecked(IDC_CHECK1) ? 1 : 0;
			m_regkey.SetDWORDValue(L"bitdepth", bCheck);
			Explorecam_put_Option(m_hCam, EXPLORE_OPTION_BITDEPTH, bCheck);
			if (bCheck)
			{
				m_scale = IsDlgButtonChecked(IDC_CHECK2) ? 1 : 0;
				m_regkey.SetDWORDValue(L"scale", m_scale);
			}
		}
		{
			CComboBox box(GetDlgItem(IDC_COMBO1));
			m_regkey.SetDWORDValue(L"trigger", box.GetCurSel());
			if (0 == box.GetCurSel())
				Explorecam_put_Option(m_hCam, EXPLORE_OPTION_TRIGGER, 1);
		}

		EndDialog(wID);
		return 0;
	}

	LRESULT OnCancel(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		EndDialog(wID);
		return 0;
	}

	LRESULT OnAdd(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		CExposureDlg dlg(m_expoTimeRange, m_expoGainRange);
		if (IDOK == dlg.DoModal())
		{
			const Expo expo = { dlg.m_delayTime, dlg.m_expoTime, dlg.m_expoGain };
			m_vecExpo.push_back(expo);

			CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
			
			ctrl.AddItem(m_vecExpo.size() - 1, 0, (LPCTSTR)FormatExpoTime(expo.expoTime));
			ctrl.SetItemText(m_vecExpo.size() - 1, 1, (LPCTSTR)FormatString(L"%hu", expo.expoGain));
			ctrl.SetItemText(m_vecExpo.size() - 1, 2, (LPCTSTR)FormatString(L"%u", expo.delayTime));

			GetDlgItem(IDOK).EnableWindow(!m_vecExpo.empty());
		}

		return 0;
	}

	LRESULT OnNmDblclk(int /*idCtrl*/, LPNMHDR pnmh, BOOL& /*bHandled*/)
	{
		NMITEMACTIVATE* pNMITEMACTIVATE = (NMITEMACTIVATE*)pnmh;
		if (pNMITEMACTIVATE->iItem >= 0)
		{
			CExposureDlg dlg(m_expoTimeRange, m_expoGainRange, m_vecExpo[pNMITEMACTIVATE->iItem]);
			if (IDOK == dlg.DoModal())
			{
				ATLASSERT(pNMITEMACTIVATE->iItem < m_vecExpo.size());
				m_vecExpo[pNMITEMACTIVATE->iItem].expoTime = dlg.m_expoTime;
				m_vecExpo[pNMITEMACTIVATE->iItem].expoGain = dlg.m_expoGain;
				m_vecExpo[pNMITEMACTIVATE->iItem].delayTime = dlg.m_delayTime;

				CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
				ctrl.SetItemText(pNMITEMACTIVATE->iItem, 0, (LPCTSTR)FormatExpoTime(dlg.m_expoTime));
				ctrl.SetItemText(pNMITEMACTIVATE->iItem, 1, (LPCTSTR)FormatString(L"%hu", dlg.m_expoGain));
				ctrl.SetItemText(pNMITEMACTIVATE->iItem, 2, (LPCTSTR)FormatString(L"%u", dlg.m_delayTime));
			}
		}
		return 0;
	}

	LRESULT OnDelete(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
		const int idx = ctrl.GetSelectedIndex();
		if (idx >= 0)
		{
			ctrl.DeleteItem(idx);
			m_vecExpo.erase(m_vecExpo.begin() + idx);

			GetDlgItem(IDOK).EnableWindow(!m_vecExpo.empty());
		}
		return 0;
	}

	LRESULT OnLvnItemchanged(int /*idCtrl*/, LPNMHDR /*pnmh*/, BOOL& /*bHandled*/)
	{
		CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
		GetDlgItem(IDC_BUTTON2).EnableWindow(ctrl.GetSelectedIndex() >= 0);
		return 0;
	}

	LRESULT OnCheck1(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (E_NOTIMPL != Explorecam_get_Option(m_hCam, EXPLORE_OPTION_BITDEPTH, nullptr)) // support bitdepth
			GetDlgItem(IDC_CHECK2).EnableWindow(IsDlgButtonChecked(IDC_CHECK1));
		return 0;
	}
};

class COffsetDlg : public CDialogImpl<COffsetDlg>
{
	friend class CSettingsDlg;
	int m_val;
public:
	enum { IDD = IDD_OFFSET };
	COffsetDlg(int val)
	: m_val(val)
	{
	}

	BEGIN_MSG_MAP(CMainDlg)
		MESSAGE_HANDLER(WM_INITDIALOG, OnInitDialog)
		COMMAND_HANDLER(IDOK, BN_CLICKED, OnOK)
		COMMAND_HANDLER(IDCANCEL, BN_CLICKED, OnCancel)
	END_MSG_MAP()
private:
	LRESULT OnInitDialog(UINT /*uMsg*/, WPARAM /*wParam*/, LPARAM /*lParam*/, BOOL& /*bHandled*/)
	{
		CenterWindow(GetParent());

		SetDlgItemInt(IDC_EDIT1, m_val);
		return TRUE;
	}

	LRESULT OnOK(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		BOOL bTrans = FALSE;
		m_val = GetDlgItemInt(IDC_EDIT1, &bTrans, TRUE);
		if (!bTrans)
		{
			GotoDlgCtrl(GetDlgItem(IDC_EDIT1));
			AtlMessageBox(m_hWnd, L"Format error.", (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
			return true;
		}

		EndDialog(wID);
		return 0;
	}

	LRESULT OnCancel(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		EndDialog(wID);
		return 0;
	}
};

class CSettingsDlg : public CDialogImpl<CSettingsDlg>
{
	const int m_row, m_col;
	std::vector<bool>& m_vecVisible;
	std::vector<int>& m_vecOffset;
public:
	enum { IDD = IDD_SETTINGS };
	CSettingsDlg(std::vector<bool>& vecVisible, std::vector<int>& vecOffset, int row, int col)
	: m_row(row), m_col(col), m_vecVisible(vecVisible), m_vecOffset(vecOffset)
	{
	}

	BEGIN_MSG_MAP(CMainDlg)
		MESSAGE_HANDLER(WM_INITDIALOG, OnInitDialog)
		COMMAND_HANDLER(IDOK, BN_CLICKED, OnOK)
		COMMAND_HANDLER(IDCANCEL, BN_CLICKED, OnCancel)
		COMMAND_HANDLER(IDC_BUTTON1, BN_CLICKED, OnOffset)
		COMMAND_HANDLER(IDC_BUTTON2, BN_CLICKED, OnUnselectAll)
		COMMAND_HANDLER(IDC_BUTTON3, BN_CLICKED, OnAutoOffset)
		NOTIFY_HANDLER(IDC_LIST1, LVN_ITEMCHANGED, OnLvnItemchanged)
		NOTIFY_HANDLER(IDC_LIST1, NM_DBLCLK, OnNmDblclk)
	END_MSG_MAP()
private:
	LRESULT OnInitDialog(UINT /*uMsg*/, WPARAM /*wParam*/, LPARAM /*lParam*/, BOOL& /*bHandled*/)
	{
		CenterWindow(GetParent());

		CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
		ctrl.SetExtendedListViewStyle(ctrl.GetExtendedListViewStyle() | LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_CHECKBOXES);
		ctrl.AddColumn(L"Zone", 0);
		ctrl.AddColumn(L"Offset", 1);
		CRect rect;
		ctrl.GetClientRect(&rect);
		const int width = rect.Width() - GetSystemMetrics(SM_CXVSCROLL) - 8;
		ctrl.SetColumnWidth(0, width * 2 / 3);
		ctrl.SetColumnWidth(1, width / 3);
		for (int j = 0; j < m_row; ++j)
		{
			for (int i = 0; i < m_col; ++i)
			{
				ctrl.AddItem(j * m_col + i, 0, (LPCTSTR)FormatString(L"(%d, %d)", j + 1, i + 1));
				ctrl.SetItemText(j * m_col + i, 1, (LPCTSTR)FormatString(L"%d", m_vecOffset[j * m_col + i]));
				ctrl.SetCheckState(j * m_col + i, m_vecVisible[j * m_col + i]);
			}
		}

		OnCheckState();
		GetDlgItem(IDC_BUTTON1).EnableWindow(FALSE);
		return TRUE;
	}

	LRESULT OnUnselectAll(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
		for (int i = 0; i < m_row * m_col; ++i)
			ctrl.SetCheckState(i, FALSE);
		GetDlgItem(IDOK).EnableWindow(FALSE);
		GetDlgItem(IDC_BUTTON3).EnableWindow(FALSE);
		return 0;
	}

	LRESULT OnOffset(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
		OffsetDlg(ctrl.GetSelectedIndex());
		return 0;
	}

	LRESULT OnAutoOffset(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		COffsetDlg dlg(0);
		if (IDOK == dlg.DoModal())
		{
			CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
			for (int i = 0, j = 0; i < m_row * m_col; ++i)
			{
				if (ctrl.GetCheckState(i))
				{
					m_vecOffset[i] = dlg.m_val * (j++);
					ctrl.SetItemText(i, 1, (LPCTSTR)FormatString(L"%d", m_vecOffset[i]));
				}
			}
		}
		return 0;
	}

	LRESULT OnNmDblclk(int /*idCtrl*/, LPNMHDR pnmh, BOOL& /*bHandled*/)
	{
		OffsetDlg(((NMITEMACTIVATE*)pnmh)->iItem);
		return 0;
	}

	LRESULT OnLvnItemchanged(int /*idCtrl*/, LPNMHDR /*pnmh*/, BOOL& /*bHandled*/)
	{
		CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
		GetDlgItem(IDC_BUTTON1).EnableWindow(ctrl.GetSelectedIndex() >= 0);
		OnCheckState();
		return 0;
	}

	LRESULT OnOK(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
		for (int i = 0; i < m_row * m_col; ++i)
			m_vecVisible[i] = ctrl.GetCheckState(i) ? true : false;

		EndDialog(wID);
		return 0;
	}

	LRESULT OnCancel(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		EndDialog(wID);
		return 0;
	}
private:
	void OffsetDlg(const int idx)
	{
		if (idx >= 0)
		{
			ATLASSERT(idx < m_row * m_col);

			COffsetDlg dlg(m_vecOffset[idx]);
			if (IDOK == dlg.DoModal())
			{
				m_vecOffset[idx] = dlg.m_val;

				CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
				ctrl.SetItemText(idx, 1, (LPCTSTR)FormatString(L"%d", m_vecOffset[idx]));
			}
		}
	}

	void OnCheckState()
	{
		CListViewCtrl ctrl(GetDlgItem(IDC_LIST1));
		bool bHasSelect = false;
		for (int i = 0; i < m_row * m_col; ++i)
		{
			if (ctrl.GetCheckState(i))
			{
				bHasSelect = true;
				break;
			}
		}
		GetDlgItem(IDOK).EnableWindow(bHasSelect);
		GetDlgItem(IDC_BUTTON3).EnableWindow(bHasSelect);
	}
};

class CVideoView : public CWindowImpl<CVideoView>
{
	BITMAPINFOHEADER m_header;
	unsigned char m_nBitDepth;
	unsigned m_nBayer;
	void* m_pRgbData;

	BEGIN_MSG_MAP(CVideoView)
		MESSAGE_HANDLER(WM_PAINT, OnWmPaint)
		MESSAGE_HANDLER(WM_ERASEBKGND, OnEraseBkgnd)
	END_MSG_MAP()
	static ATL::CWndClassInfo& GetWndClassInfo()
	{
		static ATL::CWndClassInfo wc =
		{
			{ sizeof(WNDCLASSEX), CS_HREDRAW | CS_VREDRAW, StartWindowProc,
			  0, 0, NULL, NULL, NULL, (HBRUSH)NULL_BRUSH, NULL, NULL, NULL },
			NULL, NULL, IDC_ARROW, TRUE, 0, L""
		};
		return wc;
	}
public:
	CVideoView()
		: m_pRgbData(nullptr), m_nBayer(0), m_nBitDepth(0)
	{
		memset(&m_header, 0, sizeof(m_header));
		m_header.biSize = sizeof(m_header);
		m_header.biBitCount = 24;
		m_header.biPlanes = 1;
	}

	virtual ~CVideoView()
	{
		if (m_pRgbData)
		{
			free(m_pRgbData);
			m_pRgbData = nullptr;
		}
	}

	void Init(LONG w, LONG h, unsigned nBayer, unsigned char nBitDepth)
	{
		m_nBayer = nBayer;
		m_nBitDepth = nBitDepth;
		m_header.biWidth = w;
		m_header.biHeight = h;
		m_header.biSizeImage = TDIBWIDTHBYTES(w * m_header.biBitCount) * h;
		if (m_pRgbData)
		{
			free(m_pRgbData);
			m_pRgbData = nullptr;
		}
		m_pRgbData = malloc(m_header.biSizeImage);
	}

	void SetData(void* pRawData)
	{
		Explorecam_deBayerV2(m_nBayer, m_header.biWidth, m_header.biHeight, pRawData, m_pRgbData, m_nBitDepth, m_header.biBitCount);
		Invalidate(FALSE);
	}
private:
	LRESULT OnEraseBkgnd(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) { return 1; }
	LRESULT OnWmPaint(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
	{
		CPaintDC dc(m_hWnd);

		CRect rc, newrc;
		GetClientRect(&rc);
		if (m_pRgbData)
		{
			const double dx = ((double)rc.Width()) / m_header.biWidth;
			const double dy = ((double)rc.Height()) / m_header.biHeight;
			const double dxy = __min(dx, dy);
			const LONG neww = (LONG)(dxy * m_header.biWidth);
			const LONG newh = (LONG)(dxy * m_header.biHeight);
			if (rc.Width() > neww)
			{
				newrc.SetRect(rc.left, rc.top, (rc.Width() - neww) / 2, rc.bottom);
				dc.FillRect(&newrc, GetSysColor(COLOR_BTNFACE));
				newrc.SetRect(neww + (rc.Width() - neww) / 2, rc.top, rc.right, rc.bottom);
				dc.FillRect(&newrc, GetSysColor(COLOR_BTNFACE));
			}
			if (rc.Height() > newh)
			{
				newrc.SetRect(rc.left, rc.top, rc.right, (rc.Height() - newh) / 2);
				dc.FillRect(&newrc, GetSysColor(COLOR_BTNFACE));
				newrc.SetRect(rc.left, newh + (rc.Height() - newh) / 2, rc.right, rc.bottom);
				dc.FillRect(&newrc, GetSysColor(COLOR_BTNFACE));
			}
			const int m = dc.SetStretchBltMode(COLORONCOLOR);
			StretchDIBits(dc, (rc.Width() - neww) / 2, (rc.Height() - newh) / 2, neww, newh, 0, 0, m_header.biWidth, m_header.biHeight, m_pRgbData, (BITMAPINFO*)&m_header, DIB_RGB_COLORS, SRCCOPY);
			dc.SetStretchBltMode(m);
		}
		else
		{
			dc.FillSolidRect(&rc, GetSysColor(COLOR_BTNFACE));
		}

		return 0;
	}
};

class CMainFrame : public CFrameWindowImpl<CMainFrame>, public CUpdateUI<CMainFrame>, public CIdleHandler
{
	HExplorecam		m_hCam;
	ExplorecamDeviceV2	m_dev;
	void*			m_pRawData;
	int				m_ymax;
	int				m_scale;
	unsigned		m_row, m_col, m_area, m_bitdepth;
	DWORD			m_tickLast;
	bool			m_bTriggerMode, m_bWantTigger, m_bTemperature;
	CVideoView		m_view;
	CTabCtrl		m_tabCtrl;
	CGraph			m_tempGraph; //temperature
	CGraph*			m_curGraph;
	CWindow*		m_curWnd;
	int				m_idxExpo;
	std::vector<CGraph>	m_vecGraph;
	std::vector<Expo>	m_vecExpo;
public:
	CMainFrame()
	: m_hCam(nullptr), m_pRawData(nullptr), m_curGraph(nullptr), m_curWnd(nullptr), m_idxExpo(-1)
	, m_bTriggerMode(false), m_bWantTigger(false), m_bTemperature(false), m_ymax(0), m_bitdepth(0), m_scale(1), m_row(10), m_col(10), m_area(5)
	, m_tempGraph(true)
	{
	}

	BEGIN_MSG_MAP(CMainFrame)
		MSG_WM_CREATE(OnCreate)
		MSG_WM_TIMER(OnTimer)
		MESSAGE_HANDLER(WM_DESTROY, OnWmDestroy)
		MESSAGE_HANDLER(MSG_CAMERA, OnMsgCamera)
		COMMAND_ID_HANDLER(ID_BESTFIT, OnBestfit)
		COMMAND_ID_HANDLER(ID_ZOOMIN_X, OnZoominX)
		COMMAND_ID_HANDLER(ID_ZOOMOUT_X, OnZoomoutX)
		COMMAND_ID_HANDLER(ID_ZOOMIN_Y, OnZoominY)
		COMMAND_ID_HANDLER(ID_ZOOMOUT_Y, OnZoomoutY)
		COMMAND_ID_HANDLER(ID_COPY, OnCopy)
		COMMAND_ID_HANDLER(ID_CSV, OnCsv)
		COMMAND_ID_HANDLER(ID_START, OnStart)
		COMMAND_ID_HANDLER(ID_STOP, OnStop)
		COMMAND_ID_HANDLER(ID_SETTINGS, OnSettings)
		NOTIFY_CODE_HANDLER(TTN_GETDISPINFOW, OnToolTipText)
		NOTIFY_HANDLER(1, TCN_SELCHANGE, OnTcnSelChange)
		CHAIN_MSG_MAP(CUpdateUI<CMainFrame>)
		CHAIN_MSG_MAP(CFrameWindowImpl<CMainFrame>)
	END_MSG_MAP()

	DECLARE_FRAME_WND_CLASS(NULL, IDR_MAIN);

	BEGIN_UPDATE_UI_MAP(CMainFrame)
		UPDATE_ELEMENT(ID_BESTFIT, UPDUI_TOOLBAR)
		UPDATE_ELEMENT(ID_ZOOMIN_X, UPDUI_TOOLBAR)
		UPDATE_ELEMENT(ID_ZOOMOUT_X, UPDUI_TOOLBAR)
		UPDATE_ELEMENT(ID_ZOOMIN_Y, UPDUI_TOOLBAR)
		UPDATE_ELEMENT(ID_ZOOMOUT_Y, UPDUI_TOOLBAR)
		UPDATE_ELEMENT(ID_CSV, UPDUI_TOOLBAR)
		UPDATE_ELEMENT(ID_SETTINGS, UPDUI_TOOLBAR)
		UPDATE_ELEMENT(ID_START, UPDUI_TOOLBAR)
		UPDATE_ELEMENT(ID_STOP, UPDUI_TOOLBAR)
	END_UPDATE_UI_MAP()
public:
	virtual BOOL OnIdle()
	{
		UIEnable(ID_START, nullptr == m_hCam);
		UIEnable(ID_STOP, m_hCam ? TRUE : FALSE);
		UIEnable(ID_BESTFIT, m_curGraph && (m_curGraph->dataNum() > 0));
		UIEnable(ID_ZOOMIN_X, m_curGraph && (m_curGraph->CanZoomIn(true)));
		UIEnable(ID_ZOOMOUT_X, m_curGraph && (m_curGraph->CanZoomOut(true)));
		UIEnable(ID_ZOOMIN_Y, m_curGraph && (m_curGraph->CanZoomIn(false)));
		UIEnable(ID_ZOOMOUT_Y, m_curGraph && (m_curGraph->CanZoomOut(false)));
		UIEnable(ID_CSV, m_curGraph && (m_curGraph->dataNum() > 0));
		UIEnable(ID_SETTINGS, m_hCam ? TRUE : FALSE);
		UIUpdateToolBar();
		return FALSE;
	}

	int OnCreate(LPCREATESTRUCT /*lpCreateStruct*/)
	{
		CreateSimpleToolBar(IDR_TOOLBAR);		
		UIAddToolBar(m_hWndToolBar);
		m_hWndClient = m_tabCtrl.Create(m_hWnd, nullptr, nullptr, WS_CHILD | WS_VISIBLE, 0, 1);
		m_tempGraph.Create(m_tabCtrl.m_hWnd, nullptr, nullptr, WS_CHILD);
		m_view.Create(m_tabCtrl.m_hWnd, nullptr, nullptr, WS_CHILD);
		CenterWindow(GetParent());

		CMessageLoop* pLoop = _Module.GetMessageLoop();
		pLoop->AddIdleHandler(this);
		return TRUE;
	}

	LRESULT OnWmDestroy(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
	{
		CloseCamera();
		CFrameWindowImpl<CMainFrame>::OnDestroy(uMsg, wParam, lParam, bHandled);
		return 0;
	}

	LRESULT OnToolTipText(int idCtrl, LPNMHDR pnmh, BOOL& /*bHandled*/)
	{
		LPNMTTDISPINFOW pDispInfo = (LPNMTTDISPINFOW)pnmh;
		if (idCtrl && !(pDispInfo->uFlags & TTF_IDISHWND))
		{
			static constexpr struct {
				int id;
				const wchar_t* str;
			}arr[] = {
				{ ID_BESTFIT, L"Best fit" },
				{ ID_ZOOMIN_X, L"X Zoom in" },
				{ ID_ZOOMOUT_X, L"X Zoom out" },
				{ ID_ZOOMIN_Y, L"Y Zoom in" },
				{ ID_ZOOMOUT_Y, L"Y Zoom out" },
				{ ID_COPY, L"Copy to clipboard" },
				{ ID_CSV, L"Export data to csv file" },
				{ ID_START, L"Start" },
				{ ID_STOP, L"Stop" },
				{ ID_SETTINGS, L"Settings" }
			};
			for (size_t i = 0; i < _countof(arr); ++i)
			{
				if (idCtrl == arr[i].id)
				{
					wcscpy(pDispInfo->szText, arr[i].str);
					pDispInfo->uFlags |= TTF_DI_SETITEM;
					break;
				}
			}
		}

		return 0;
	}

	LRESULT OnSettings(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (m_hCam && m_vecGraph.size())
		{
			std::vector<bool> vecVisible(m_row * m_col);
			std::vector<int> vecOffset(m_row * m_col);
			m_vecGraph[0].Get(vecVisible, vecOffset);
			CSettingsDlg dlg(vecVisible, vecOffset, m_row, m_col);
			if (IDOK == dlg.DoModal())
			{
				for (size_t i = 0; i < m_vecGraph.size(); ++i)
					m_vecGraph[i].Set(vecVisible, vecOffset);
			}
		}
		return 0;
	}

	LRESULT OnBestfit(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (m_curGraph)
			m_curGraph->Zoom11();
		return 0;
	}

	LRESULT OnZoominX(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (m_curGraph)
			m_curGraph->ZoomIn(true);
		return 0;
	}

	LRESULT OnZoomoutX(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (m_curGraph)
			m_curGraph->ZoomOut(true);
		return 0;
	}

	LRESULT OnZoominY(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (m_curGraph)
			m_curGraph->ZoomIn(false);
		return 0;
	}

	LRESULT OnZoomoutY(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (m_curGraph)
			m_curGraph->ZoomOut(false);
		return 0;
	}

	LRESULT OnCopy(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (m_curGraph)
		{
			PBYTE pdib = m_curGraph->GetBitmap();
			if (pdib)
			{
				if (OpenClipboard())
				{
					EmptyClipboard();
					const PBITMAPINFOHEADER h = (const PBITMAPINFOHEADER)pdib;
					const DWORD dwLen = h->biSize + TDIBWIDTHBYTES(h->biWidth * h->biBitCount) * h->biHeight;
					HANDLE hCopy = GlobalAlloc(GMEM_MOVEABLE, dwLen);
					if (hCopy)
					{
						void* pCopy = GlobalLock(hCopy);
						if (nullptr == pCopy)
							GlobalFree(hCopy);
						else
						{
							memcpy(pCopy, h, dwLen);
							GlobalUnlock(hCopy);

							SetClipboardData(CF_DIB, hCopy);
							CloseClipboard();
						}
					}
				}

				free(pdib);
			}
		}
		return 0;
	}

	LRESULT OnCsv(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (m_curGraph && (m_curGraph->dataNum() > 0))
		{
			CFileDialog dlg(FALSE);
			if (IDOK == dlg.DoModal())
			{
				wchar_t str[MAX_PATH];
				for (int i = 0; i < m_vecExpo.size(); ++i)
				{
					swprintf(str, L"%s-%sms-%u.csv", dlg.m_szFileName, (LPCTSTR)FormatExpoTime(m_vecExpo[i].expoTime), m_vecExpo[i].expoGain);
					m_vecGraph[i].OnCsv(str);
				}
				if (m_bTemperature && (m_tempGraph.dataNum() > 0))
				{
					swprintf(str, L"%s-Temperature.csv", dlg.m_szFileName);
					m_tempGraph.OnCsv(str);
				}
			}
		}
		return 0;
	}
	
	LRESULT OnStop(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		CloseCamera();
		return 0;
	}

	LRESULT OnStart(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL& /*bHandled*/)
	{
		if (nullptr == m_hCam)
		{
			ExplorecamDeviceV2 arr[EXPLORE_MAX];
			const unsigned num = Explorecam_EnumV2(arr);
			if (num <= 0)
				AtlMessageBox(m_hWnd, L"No camera found.", (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
			else if (1 == num)
				OpenCamera(arr[0]);
			else
			{
				CPoint pt;
				GetCursorPos(&pt);
				CMenu menu;
				menu.CreatePopupMenu();
				for (unsigned i = 0; i < num; ++i)
					menu.AppendMenu(MF_STRING, ID_CAMERA00 + i, arr[i].displayname);
				const int ret = menu.TrackPopupMenu(TPM_RIGHTALIGN | TPM_RETURNCMD, pt.x, pt.y, m_hWnd);
				if (ret >= ID_CAMERA00)
				{
					ATLASSERT(ret - ID_CAMERA00 < num);
					OpenCamera(arr[ret - ID_CAMERA00]);
				}
			}
		}
		return 0;
	}

	LRESULT OnMsgCamera(UINT /*uMsg*/, WPARAM wParam, LPARAM /*lParam*/, BOOL& /*bHandled*/)
	{
		switch (wParam)
		{
		case EXPLORE_EVENT_ERROR:
			CloseCamera();
			AtlMessageBox(m_hWnd, L"Generic error.", (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
			break;
		case EXPLORE_EVENT_DISCONNECTED:
			CloseCamera();
			AtlMessageBox(m_hWnd, L"Camera disconnect.", (LPCTSTR)nullptr, MB_OK | MB_ICONWARNING);
			break;
		case EXPLORE_EVENT_IMAGE:
			OnEventImage();
			break;
		default:
			break;
		}
		return 0;
	}

	void OnTimer(UINT_PTR nIDEvent)
	{
		switch (nIDEvent)
		{
		case TIMER_ID:
			if (m_bWantTigger && (GetTickCount() - m_tickLast >= m_vecExpo[m_idxExpo].delayTime + TIMER_EPSILON))
			{
				m_bWantTigger = false;
				Explorecam_Trigger(m_hCam, 1);
			}
			break;
		case TIMER_TEMP:
			if (m_bTemperature)
			{
				short temp = 0;
				if (SUCCEEDED(Explorecam_get_Temperature(m_hCam, &temp)))
				{
					int val = temp;
					m_tempGraph.AddData(&val);
				}
			}
			break;
		default:
			break;
		}
	}

	LRESULT OnTcnSelChange(int /*idCtrl*/, LPNMHDR /*pnmh*/, BOOL& /*bHandled*/)
	{
		const int idx = m_tabCtrl.GetCurSel();
		CWindow* newWnd = nullptr;
		if (idx + 1 == m_tabCtrl.GetItemCount())
			newWnd = &m_view;
		else if (m_bTemperature && (idx + 2 == m_tabCtrl.GetItemCount()))
			newWnd = &m_tempGraph;
		else if ((idx >= 0) && (idx < m_vecGraph.size()))
			newWnd = &m_vecGraph[idx];
		if (newWnd && (m_curWnd != newWnd))
		{
			if (m_curWnd)
				m_curWnd->ShowWindow(SW_HIDE);
			m_curWnd = newWnd;
			if ((idx >= 0) && (idx < m_vecGraph.size()))
				m_curGraph = &m_vecGraph[idx];
			else
				m_curGraph = nullptr;
			m_curWnd->ShowWindow(SW_SHOW);
			UpdateLayout();
		}
		return 0;
	}

	void UpdateLayout(BOOL bResizeBars = TRUE)
	{
		CFrameWindowImpl<CMainFrame>::UpdateLayout(bResizeBars);
		
		CRect rect;
		m_tabCtrl.GetClientRect(&rect);
		m_tabCtrl.AdjustRect(FALSE, &rect);
		for (size_t i = 0; i < m_vecExpo.size(); ++i)
			m_vecGraph[i].MoveWindow(&rect);
		m_tempGraph.MoveWindow(&rect);
		m_view.MoveWindow(&rect);
	}
private:
	void OpenCamera(const ExplorecamDeviceV2& dev)
	{
		m_dev = dev;
		m_hCam = Explorecam_Open(m_dev.id);
		Explorecam_put_eSize(m_hCam, 0); // always use the maximum resolution
		Explorecam_put_Option(m_hCam, EXPLORE_OPTION_RAW, 1);
		Explorecam_put_HZ(m_hCam, 2);
		Explorecam_put_AutoExpoEnable(m_hCam, 0); // always disable auto exposure

		CConfigDlg dlg(m_hCam);
		if (IDOK != dlg.DoModal())
			CloseCamera();
		else
		{
			while (m_tabCtrl.GetItemCount())
				m_tabCtrl.DeleteItem(0);
			for (size_t i = 0; i < m_vecGraph.size(); ++i)
				m_vecGraph[i].DestroyWindow();
			m_vecGraph.clear();

			m_row = dlg.m_row;
			m_col = dlg.m_col;
			m_area = dlg.m_area;
			m_vecExpo = dlg.m_vecExpo;
			m_scale = 1;

			unsigned nFourCC;
			Explorecam_get_RawFormat(m_hCam, &nFourCC, &m_bitdepth);
			m_ymax = 1 << m_bitdepth;
			if (dlg.m_bin & 0x40)
				m_ymax *= (dlg.m_bin & 0xf) * (dlg.m_bin & 0xf);
			if ((m_bitdepth > 8) && dlg.m_scale && (m_ymax < 65536))
			{
				m_scale = 65536 / m_ymax;
				m_ymax = 65536;
			}
			
			int w = 0, h = 0;
			Explorecam_get_FinalSize(m_hCam, &w, &h);
			m_pRawData = malloc(w * h * ((m_bitdepth > 8) ? 2 : 1));

			m_tempGraph.Init(0, 0);
			m_view.Init(w, h, nFourCC, m_bitdepth);
			m_vecGraph.resize(m_vecExpo.size());
			for (size_t i = 0; i < m_vecExpo.size(); ++i)
			{
				m_vecGraph[i].Create(m_tabCtrl.m_hWnd, nullptr, nullptr, WS_CHILD | (i ? 0 : WS_VISIBLE));
				m_vecGraph[i].Init(m_row * m_col, m_ymax);
				m_tabCtrl.AddItem((LPCTSTR)FormatString(L"%sms, %hu", (LPCTSTR)FormatExpoTime(m_vecExpo[i].expoTime), m_vecExpo[i].expoGain));
			}
			m_curGraph = &m_vecGraph[0];
			m_curWnd = m_curGraph;
			if ((E_NOTIMPL != Explorecam_get_Temperature(m_hCam, nullptr)) && dlg.m_temp)
			{
				m_tabCtrl.AddItem(L"Temperature");
				SetTimer(TIMER_TEMP, dlg.m_temp * 1000, nullptr);
				m_bTemperature = true;
			}
			else
			{
				m_bTemperature = false;
			}
			m_tabCtrl.AddItem(L"Video");

			int val = 0;
			Explorecam_get_Option(m_hCam, EXPLORE_OPTION_TRIGGER, &val);
			m_bTriggerMode = (val != 0);
			
			m_idxExpo = 0;
			Explorecam_put_ExpoTime(m_hCam, m_vecExpo[m_idxExpo].expoTime);
			Explorecam_put_ExpoAGain(m_hCam, m_vecExpo[m_idxExpo].expoGain);
			Explorecam_StartPullModeWithWndMsg(m_hCam, m_hWnd, MSG_CAMERA);
			if (m_bTriggerMode)
			{
				Explorecam_Trigger(m_hCam, 1);
				m_bWantTigger = false;
			}
			SetTimer(TIMER_ID, TIMER_EPSILON, nullptr);
			m_tickLast = GetTickCount();

			UpdateLayout();
		}
	}

	void CloseCamera()
	{
		if (m_hCam)
		{
			Explorecam_Close(m_hCam);
			m_hCam = nullptr;
		}
		if (m_pRawData)
		{
			free(m_pRawData);
			m_pRawData = nullptr;
		}
		KillTimer(TIMER_ID);
		KillTimer(TIMER_TEMP);
	}

	template<typename T>
	void GetData(int* arr, const T* pData, const ExplorecamFrameInfoV3& info)
	{
		const unsigned divx = info.width / m_col, divy = info.height / m_row;

		for (int j = 0; j < m_row; ++j)
		{
			for (int i = 0; i < m_col; ++i)
			{
				long long sum = 0;
				for (int y = 0; y < m_area; ++y)
				{
					const T* p = pData + info.width * (j * divy + y + (info.height - divy * m_row) / 2) + (i * divx + (info.width - divx * m_col) / 2);
					for (int x = 0; x < m_area; ++x)
						sum += p[x];
				}
				arr[m_col * j + i] = sum * m_scale / (m_area * m_area);
			}
		}
	}

	void OnEventImage()
	{
		const DWORD dwTick = GetTickCount();
		bool bAdd = false;
		ExplorecamFrameInfoV3 info = { 0 };
		const HRESULT hr = Explorecam_PullImageV3(m_hCam, m_pRawData, 0, 0, 0, &info);
		if (SUCCEEDED(hr))
		{
			if (m_bTriggerMode)
				bAdd = true;
			else if (dwTick - m_tickLast >= m_vecExpo[m_idxExpo].delayTime + TIMER_EPSILON)
				bAdd = true;
			if (bAdd)
			{
				int* arr = (int*)alloca(sizeof(int) * m_row * m_col);
				if (m_bitdepth > 8)
					GetData(arr, (const PUSHORT)m_pRawData, info);
				else
					GetData(arr, (const PBYTE)m_pRawData, info);
				m_vecGraph[m_idxExpo].AddData(arr);
			}
			m_view.SetData(m_pRawData);
		}

		if (bAdd)
		{
			m_tickLast = dwTick;

			if (m_vecExpo.size() > 1)
			{
				m_idxExpo = (++m_idxExpo) % m_vecExpo.size();
				Explorecam_put_ExpoTime(m_hCam, m_vecExpo[m_idxExpo].expoTime);
				Explorecam_put_ExpoAGain(m_hCam, m_vecExpo[m_idxExpo].expoGain);
			}
			if (m_bTriggerMode)
			{
				if (0 == m_vecExpo[m_idxExpo].delayTime)
					Explorecam_Trigger(m_hCam, 1);
				else
					m_bWantTigger = true;
			}
		}
	}
};

static int Run()
{
	CMessageLoop theLoop;
	_Module.AddMessageLoop(&theLoop);

	g_curHand1 = AtlLoadCursor(IDC_HAND1);
	g_curHand2 = AtlLoadCursor(IDC_HAND2);
	g_curMove = AtlLoadCursor(IDC_MOVE);
	g_curArrow = LoadCursor(nullptr, MAKEINTRESOURCE(IDC_ARROW));

	CMainFrame frmMain;
	if (frmMain.CreateEx() == nullptr)
		return 0;
	frmMain.ShowWindow(SW_SHOWMAXIMIZED);

	int nRet = theLoop.Run();
	_Module.RemoveMessageLoop();
	return nRet;
}

int WINAPI _tWinMain(HINSTANCE hInstance, HINSTANCE /*hPrevInstance*/, LPTSTR /*pCmdLine*/, int /*nCmdShow*/)
{
	INITCOMMONCONTROLSEX iccx;
	iccx.dwSize = sizeof(iccx);
	iccx.dwICC = ICC_COOL_CLASSES | ICC_BAR_CLASSES;
	InitCommonControlsEx(&iccx);
	OleInitialize(nullptr);
	_Module.Init(nullptr, hInstance);
	int nRet = Run();
	_Module.Term();
	return nRet;
}

#if defined _M_IX86
#pragma comment(linker, "/manifestdependency:\"type='win32' name='Microsoft.Windows.Common-Controls' version='6.0.0.0' processorArchitecture='x86' publicKeyToken='6595b64144ccf1df' language='*'\"")
#elif defined _M_X64
#pragma comment(linker, "/manifestdependency:\"type='win32' name='Microsoft.Windows.Common-Controls' version='6.0.0.0' processorArchitecture='amd64' publicKeyToken='6595b64144ccf1df' language='*'\"")
#endif